/******************************************************************************/
#include "stdafx.h"
#include <EsenthelEngine/EsenthelEngine.h>
/******************************************************************************

   In order to take advantage of multiple cores on your CPU, multi-threaded rendering is used.

   It works by rendering graphics on one thread, and processing data on the other.

   To avoid multi-threaded issues you need to store your game data in two versions:
      -backuped data (which contains only data required for rendering, like position and mesh)
      -normal   data (which contains all the data of your object, including position, mesh and other custom parameters)

   Multi-threaded rendering will look like this:

   +-------------------------+-----------------------+
   |        Thread 1         |        Thread 2       |
   +-------------------------+-----------------------+
   |    Backup your data     |                       |
   +-------------------------+-----------------------+
   | Process the data update |  Render 3D graphics   |
   +-------------------------+-----------------------+
   |   Wait until rendered   |                       |
   +-------------------------+-----------------------+
   |     Draw 2D graphics    |                       |
   +-------------------------+-----------------------+

   While Rendering 3D graphics on other thread:
      - you must not modify 'Viewport' viewport or activate another viewport
      - you must not modify 'Cam'      camera   or activate another camera
      - you must not call methods changing rendering matrixes/velocities (MatrixSet, VelSet, AngVelSet, CSkeleton::setMatrix)

   Note:
      Performance gains may vary depending on:
         -your system (mainly GPU & CPU - number of cores and general performance)
         -amount of calculations in update
         -amount of objects rendered

/******************************************************************************/
struct ObjectBackup // game object backup - contains only data required for rendering
{
   Vec   pos;
   Mesh *mesh;

   void draw() // rendering method
   {
      mesh->draw(Matrix(pos)); // render 'mesh' at 'pos' position
   }
};
struct Object // game object
{
   Flt   parameters[256];
   Vec   pos;
   Mesh *mesh;

   // note that this structure doesn't contain any rendering method, as we'll be rendering only through ObjectBackup::draw

   void backup(ObjectBackup &dst) // backup data method
   {
      // backup all data needed for rendering here
      dst.pos =pos;
      dst.mesh=mesh;
   }

   Object(){Zero(T);}
};
/******************************************************************************/
Bool multi_threaded; // if multi-threaded mode

Mesh mbox , // mesh box
     mball; // mesh sphere

Memb<Object      > mobj    ; // Object's       memory block
Memb<ObjectBackup> mobj_bkp; // ObjectBackup's memory block
/******************************************************************************/
void Render()
{
   switch(Renderer())
   {
      case RM_EARLY_Z:
      case RM_SHD_MAP:
      case RM_SOLID  :
         mbox.draw(MatrixIdentity);
         REPA(mobj_bkp)mobj_bkp[i].draw(); // render backuped data
      break;

      case RM_LIGHT:
         LightPoint(1,Vec(0,0.8,0)).add();
      break;
   }
}
void Render3D()
{
   Renderer(Render);
}
void Draw2D()
{
   D.text(0,0.9,multi_threaded ? "Multi-threaded rendering" : "Single-threaded rendering");
   D.text(0,0.8,S+"CPU Cores "+Cpu.cores());
   D.text(0,0.7,S+"Fps "      +Tm.fps     );
   D.text(0,0.6,  "Press 1,2 to change rendering mode");
}
/******************************************************************************/
void InitPre()
{
   App.name="Multi-Threaded Rendering";
   App.flag=APP_MS_EXCLUSIVE|APP_FULL_TOGGLE;
   PakAdd("../data/engine.pak",NULL);

   D.full(true).bumpMode(BUMP_PARALLAX).shdSoft(2);
}
/******************************************************************************/
Bool Init()
{
   // create materials
   Material *brick=Materials("../data/mtrl/brick/0.mtrl");
   
   // create meshes
   mbox .create(1).B(0).create(Box (1   ),VTX_TX0|VTX_NRM|VTX_TNG).reverse(); // create mesh box
   mball.create(1).B(0).create(Ball(0.08),VTX_TX0|VTX_NRM|VTX_TNG)          ; // create mesh sphere

   // set mesh materials, rendering versions and bounding boxes
   mbox .setMaterial(brick).setRender().setBox();
   mball.setMaterial(brick).setRender().setBox();

   // create some objects
   REP(256)
   {
      Object &obj=mobj.New(); // create new object in container
      obj.pos =Rnd(mbox.box); // set it's random position
      obj.mesh=&mball;        // set it's mesh
   }

   if(Cpu.cores()>1) // if we have more than one core
   {
      StateMain.threadedSet(Render3D); // set StateMain (our current state) to have multi-threaded rendering
      multi_threaded=true;
   }

   return true;
}
/******************************************************************************/
void Shut()
{
}
/******************************************************************************/
Bool Main()
{
   if(Kb.bp(KB_ESC))return false;
   CamHandle(0.01f,10,CAMH_ZOOM|(Ms.b(1)?CAMH_MOVE:CAMH_ROT));

   // toggle multi-threaded rendering when keys pressed
   if(Kb.bp(KB_1)){StateActive->threadedSet(NULL    ); multi_threaded=false;} // disable multi-threaded rendering when '1' pressed
   if(Kb.bp(KB_2)){StateActive->threadedSet(Render3D); multi_threaded=true ;} //  enable multi-threaded rendering when '2' pressed

   // 4 steps of multi-threaded processing
   {
      // 1st - Backup data
      mobj.backup(mobj_bkp); // backup all object's to object_bkp's

      // 2nd - Start multi-threaded rendering
      StateActive->threadedStart(); // after this call, Render3D will be working on the second thread

      // 3rd - Process data update here
      REPA(mobj)
      {
         Object &obj=mobj[i];
         // do some heavy floating point operations
         REPA(obj.parameters)obj.parameters[i]+=Sqrt(Sin(i*PI)*Pow(i,i*2.0)*16);
      }

      // 4th - Stop multi-threaded rendering (if it hasn't finished yet, it will wait for it)
      if(StateActive->threadedStop())
      {
         // if we're here then multi-threaded rendering is enabled, so we need to call 2D drawing manually
         Draw2D();
      }
   }

   return true;
}
/******************************************************************************/
void Draw() // if multi-threaded rendering is enabled, classic 'Draw' will not be called
{
   Render3D();
     Draw2D();

   D.text(0,-0.9,"Rendering through 'Draw'");
}
/******************************************************************************/
